import { API_KEYS, API_ENDPOINTS } from '../config/api'

interface LastFmTrack {
  name: string
  artist: string
  url: string
  image: { '#text': string, size: string }[]
}

export class LastFmService {
  private token: string | null = null

  constructor() {
    // 从本地存储获取 token
    this.token = localStorage.getItem('lastfm_token')
  }

  private async fetchFromLastFm(method: string, params: Record<string, string>) {
    const queryParams = new URLSearchParams({
      method,
      api_key: API_KEYS.LASTFM_API_KEY,
      format: 'json',
      ...params
    })

    // 如果有 token，添加到请求中
    if (this.token) {
      queryParams.append('token', this.token)
    }

    const response = await fetch(`${API_ENDPOINTS.LASTFM_BASE_URL}?${queryParams}`)
    if (!response.ok) {
      throw new Error(`Last.fm API error: ${response.statusText}`)
    }
    return response.json()
  }

  // 将情绪映射到 Last.fm 的标签
  private mapEmotionToTags(emotion: string): string[] {
    const emotionTags: Record<string, string[]> = {
      'Happy': ['happy', 'upbeat', 'feel good'],
      'Sad': ['sad', 'melancholy', 'emotional'],
      'Angry': ['angry', 'aggressive', 'intense'],
      'Relaxed': ['relaxing', 'chill', 'ambient'],
      'Excited': ['energetic', 'party', 'uplifting']
    }
    return emotionTags[emotion] || []
  }

  async getRecommendedTracks(emotion: string, limit: number = 10): Promise<LastFmTrack[]> {
    const tags = this.mapEmotionToTags(emotion)
    const tag = tags[0] // 使用第一个标签作为主要搜索条件

    try {
      const data = await this.fetchFromLastFm('tag.gettoptracks', {
        tag,
        limit: limit.toString()
      })

      return data.tracks.track.map((track: any) => ({
        name: track.name,
        artist: track.artist.name,
        url: track.url,
        image: track.image
      }))
    } catch (error) {
      console.error('Error fetching tracks from Last.fm:', error)
      throw error
    }
  }

  // 获取认证 URL
  getAuthUrl(): string {
    const callbackUrl = 'http://localhost:5173/callback'
    return `http://www.last.fm/api/auth/?api_key=${API_KEYS.LASTFM_API_KEY}&cb=${encodeURIComponent(callbackUrl)}`
  }
}
